/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jssec.android.autofillframework.autofillservice;

import android.app.assist.AssistStructure;
import android.app.assist.AssistStructure.ViewNode;
import android.util.Log;
import android.view.View;
import android.view.autofill.AutofillId;
import android.view.autofill.AutofillValue;

import java.util.Arrays;

import static android.view.View.AUTOFILL_HINT_USERNAME;

final public class StructureParser {
    private final AssistStructure mStructure;
    AutofillId mUsernameFieldId = null;
    AutofillId mPasswordFieldId = null;
    String mUsernameValue = "";
    String mPasswordValue = "";

    StructureParser(AssistStructure structure) {
        mStructure = structure;
    }

    void parseForFill() {
        int nodes = mStructure.getWindowNodeCount();
        printLog("StructureParser::parseForFill");
        for (int i = 0; i < nodes; i++) {
            AssistStructure.WindowNode node = mStructure.getWindowNodeAt(i);
            AssistStructure.ViewNode view = node.getRootViewNode();
            parse(true, view, 0);
        }
    }

    void parseForSave() {
        printLog("StructureParser::parseForSave");
        int nodes = mStructure.getWindowNodeCount();
        for (int i = 0; i < nodes; i++) {
            AssistStructure.WindowNode node = mStructure.getWindowNodeAt(i);
            AssistStructure.ViewNode view = node.getRootViewNode();
            parse(false, view, 0);
        }
    }

    void parse(boolean forFill, ViewNode viewNode, int layer) {
        dumpNode(viewNode, forFill, layer);

        if (viewNode.getAutofillHints() != null) {
            String[] filteredHints = filterSupportedHints(viewNode.getAutofillHints());
            if (filteredHints != null && filteredHints.length == 1) {
                // This is just a sample for a single 'Hints'.
                //                printLog("StructureParser::parse() filteredHints=" + filteredHints[0]);
                if (forFill) {
                    if (filteredHints[0].equals(AUTOFILL_HINT_USERNAME)) {
                        mUsernameFieldId = viewNode.getAutofillId();
                    } else if (filteredHints[0].equals(View.AUTOFILL_HINT_PASSWORD)) {
                        mPasswordFieldId = viewNode.getAutofillId();
                    }
                } else {
                    if (filteredHints[0].equals(AUTOFILL_HINT_USERNAME)) {
                        mUsernameValue = viewNode.getText().toString();
                    } else if (filteredHints[0].equals(View.AUTOFILL_HINT_PASSWORD)) {
                        mPasswordValue = viewNode.getText().toString();
                    }
                }
            }
        }
        int childrenSize = viewNode.getChildCount();
        if (childrenSize > 0) {
            for (int i = 0; i < childrenSize; i++) {
                parse(forFill, viewNode.getChildAt(i), layer+1);
            }
        }
    }
    public AutofillId getFieldId(String hint) {
        if (hint.equals(View.AUTOFILL_HINT_USERNAME)) {
            return mUsernameFieldId;
        } else if (hint.equals(View.AUTOFILL_HINT_PASSWORD)) {
            return mPasswordFieldId;
        }
        return null;
    }

    public String getValue(String hint) {
        if (hint.equals(View.AUTOFILL_HINT_USERNAME)) {
            return mUsernameValue;
        } else if (hint.equals(View.AUTOFILL_HINT_PASSWORD)) {
            return mPasswordValue;
        }
        return null;
    }


    // Check if hints contains supported hints.
    static private String[] filterSupportedHints(String[] hints) {
        if (Arrays.asList(hints).contains(AUTOFILL_HINT_USERNAME) || Arrays.asList(hints).contains(View.AUTOFILL_HINT_PASSWORD)) {
            return hints;
        }
        return new String[] {};
    }

    // Output just for debug. It should be removed at release build.
    private void dumpNode(ViewNode viewNode, boolean forFill, int layer) {
        String autofillStringValue = "N/A";
        AutofillValue value = viewNode.getAutofillValue();
        if (value != null) {
            if (value.isText()) {
                autofillStringValue = value.getTextValue() + "(text)";
            } else if (value.isList()) {
                autofillStringValue = Integer.toString(value.getListValue()) + "(list)";
            } else {
                autofillStringValue = "no type";
            }
        }
        String id = (viewNode.getAutofillId() != null ? viewNode.getAutofillId().toString() : "N/A");
        String texts = (viewNode.getText() != null ? viewNode.getText().toString() : "N/A");
        String hints = (viewNode.getAutofillHints() != null) ? viewNode.getAutofillHints()[0] : "N/A";
        int type = viewNode.getAutofillType();
        String layerSpace = "";
        for (int i=0; i<layer; i++) {
            layerSpace += "    ";
        }
        printLog("StructureParse::parse(): " + layerSpace + "forFill=" + forFill + ",id=" + id + ",type=" + type + ",values=" + autofillStringValue + ",text=" + texts + ",hints=" + hints + ",class=" + viewNode.getClassName()+",childNum="+Integer.toString(viewNode.getChildCount()));
    }

    private void printLog(String msg) {
        Log.d("JssecAutofillSample", msg);
    }
}

